/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 *
 */

import org.apache.directory.server.core.partition.Partition;
import org.apache.directory.server.core.partition.impl.btree.Index;
import org.apache.directory.server.core.partition.impl.btree.jdbm.JdbmIndex;
import org.apache.directory.server.core.partition.impl.btree.jdbm.JdbmPartition;
import org.apache.directory.shared.ldap.exception.LdapNameNotFoundException;
import org.apache.directory.shared.ldap.message.AttributeImpl;
import org.apache.directory.shared.ldap.message.AttributesImpl;
import org.apache.directory.shared.ldap.message.ModificationItemImpl;
import org.apache.directory.shared.ldap.name.LdapDN;
import static org.apache.directory.server.core.integ.IntegrationUtils.getContext;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.HashSet;
import java.util.Set;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.LdapContext;

import org.apache.directory.server.core.DefaultDirectoryService;
import org.apache.directory.server.core.DirectoryService;
import org.apache.directory.server.core.entry.DefaultServerEntry;
import org.apache.directory.server.core.entry.ServerEntry;
import org.apache.directory.server.core.integ.CiRunner;
import org.apache.directory.server.core.integ.DirectoryServiceFactory;
import org.apache.directory.server.core.integ.Level;
import org.apache.directory.server.core.integ.annotations.CleanupLevel;
import org.apache.directory.server.core.integ.annotations.Factory;
import org.junit.Test;
import org.junit.runner.RunWith;

/**
 * Test LDAP operations.
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 * @version $Rev$
 */
// First, declare that we are using the CiRunner class as an extension of JUnit
@RunWith ( CiRunner.class )

// Define the level on which we will clean the server : once per class
@CleanupLevel ( Level.CLASS )

// Declare the factory used to create the initial server
@Factory ( TestLDAPOperations.MyFactory.class )

public class TestLDAPOperations
{
    public static DirectoryService service;


    /**
     * This factory initialize the server and create the first partition
     * in which we will add/delete/modify entries.
     */
    public static class MyFactory implements DirectoryServiceFactory
    {
        /**
         * 
         */
        public DirectoryService newInstance() throws NamingException
        {
            // Initialize the server
            DirectoryService service = new DefaultDirectoryService();
            
            // Enable the changeLog interceptor
            service.getChangeLog().setEnabled( true );

            // Create my test partition
            JdbmPartition partition = new JdbmPartition();
            partition.setId( "apache" );
            partition.setSuffix( "dc=Apache,dc=Org" );

            // Add some index
            HashSet<Index> indexedAttributes = new HashSet<Index>();
            indexedAttributes.add( new JdbmIndex( "objectClass" ) );
            indexedAttributes.add( new JdbmIndex( "ou" ) );
            indexedAttributes.add( new JdbmIndex( "uid" ) );
            partition.setIndexedAttributes( indexedAttributes );

            // Dfine the top level entry
            ServerEntry serverEntry = new DefaultServerEntry( service.getRegistries(), new LdapDN( "dc=Apache,dc=Org" ) );
            serverEntry.put( "objectClass", "top", "domain", "extensibleObject" );
            serverEntry.put( "dc", "Apache" );

            // Add it to the partition
            partition.setContextEntry( serverEntry );

            Set<Partition> partitions = new HashSet<Partition>();
            partitions.add( partition );

            // Add the partition to the server
            service.setPartitions( partitions );
            
            // We are done, the server is ready to be used !
            return service;
        }
    }

    
    /**
     * Test a search operation.
     * 
     * Search the whole subtree for every entries.
     */
    @Test public void testSearch() throws NamingException
    {
        LdapContext ctxRoot = getContext( "uid=admin,ou=system", service, "dc=Apache,dc=Org" );

        SearchControls sc = new SearchControls();
        sc.setSearchScope( SearchControls.SUBTREE_SCOPE );

        NamingEnumeration<SearchResult> ne = ctxRoot.search( "", "(objectClass=*)", sc );
        assertTrue( "Search should return at least one entry.", ne.hasMore() );

        SearchResult sr = ne.next();
        assertEquals( "The entry returned should be the root entry.", "dc=Apache,dc=Org", sr.getName() );
        assertFalse( "Search should return no more entries.", ne.hasMore() );
        System.out.println( "Search operation successfull" );
    }


    /**
     * Test the Delete operation
     */
    @Test public void testDelete() throws NamingException
    {
        // First create the entry
        LdapContext ctxRoot = getContext( "uid=admin,ou=system", service, "dc=Apache,dc=Org" );

        String dn = "ou=Test";

        Attributes attributes = new AttributesImpl( true );
        Attribute attribute = new AttributeImpl( "objectClass" );
        attribute.add( "top" );
        attribute.add( "organizationalUnit" );
        attributes.put( attribute );
        attributes.put( "ou", "Test" );

        DirContext ctx = ctxRoot.createSubcontext( dn, attributes );
        assertNotNull( ctx );

        // Now delete the entry
        ctxRoot.destroySubcontext( dn );

        SearchControls sc = new SearchControls();
        sc.setSearchScope( SearchControls.OBJECT_SCOPE );

        try
        {
            ctxRoot.search( dn, "(objectClass=*)", sc );
            fail( "Search should throw exception." );
        }
        catch ( LdapNameNotFoundException e )
        {
            // ignore
        }
        
        System.out.println( "Delete operation successfull" );
    }

    
    /**
     * Test an addition of a ou=test, dc=apache, dc=org entry.
     */
    @Test public void testAdd() throws NamingException
    {
        LdapContext ctxRoot = getContext( "uid=admin,ou=system", service, "dc=Apache,dc=Org" );

        String dn = "ou=Test";

        Attributes attributes = new AttributesImpl( true );
        Attribute attribute = new AttributeImpl( "objectClass" );
        attribute.add( "top" );
        attribute.add( "organizationalUnit" );
        attributes.put( attribute );
        attributes.put( "ou", "Test" );

        DirContext ctx = ctxRoot.createSubcontext( dn, attributes );
        assertNotNull( ctx );

        // The entry has been created, now search for it
        SearchControls sc = new SearchControls();
        sc.setSearchScope( SearchControls.OBJECT_SCOPE );

        NamingEnumeration<SearchResult> ne = ctxRoot.search( dn, "(objectClass=*)", sc );
        assertTrue( "Search should return at least one entry.", ne.hasMore() );

        SearchResult sr = ne.next();
        assertEquals( "The entry returned should be the entry added earlier.", "ou=Test,dc=Apache,dc=Org", sr.getName() );
        assertFalse( "Search should return no more entries.", ne.hasMore() );
        System.out.println( "Add operation successfull" );
    }


    /**
     * Test a modify operation.
     * 
     * Here, we create again the ou=test entry, which has been created in the previous
     * test, and we don't get an error because the user has been automatically removed.
     */
    @Test public void testModify() throws NamingException
    {
        // Check that the ou=Test,dc=Apache,dc=Org does not exists anymore.
        LdapContext searchRoot = getContext( "dc=Apache,dc=Org", service, "dc=Apache,dc=Org" );

        SearchControls sc = new SearchControls();
        sc.setSearchScope( SearchControls.SUBTREE_SCOPE );

        try
        {
            searchRoot.search( "ou=Test", "(objectClass=*)", sc );
            fail();
        }
        catch ( NamingException ne )
        {
            assertTrue( true );
        }

        searchRoot.close();

        // Ok, the entry has been wiped off... Create it again
        LdapContext ctxRoot = getContext( "uid=admin,ou=system", service, "dc=Apache,dc=Org" );

        String dn = "ou=Test";
        String description = "New Value";

        Attributes attributes = new AttributesImpl( true );
        Attribute attribute = new AttributeImpl( "objectClass" );
        attribute.add( "top" );
        attribute.add( "organizationalUnit" );
        attributes.put( attribute );
        attributes.put( "ou", "Test" );
        attributes.put( "description", "Old Value" );

        DirContext ctx = ctxRoot.createSubcontext( dn, attributes );
        assertNotNull( ctx );

        // Now, modify the entry.
        ModificationItemImpl[] mods = new ModificationItemImpl[1];
        mods[0] = new ModificationItemImpl( DirContext.REPLACE_ATTRIBUTE, new AttributeImpl( "description", description ) );

        ctxRoot.modifyAttributes( dn, mods );

        sc = new SearchControls();
        sc.setSearchScope( SearchControls.OBJECT_SCOPE );

        NamingEnumeration<SearchResult> ne = ctxRoot.search( dn, "(objectClass=*)", sc );
        assertTrue( "Search should return at least one entry.", ne.hasMore() );

        SearchResult sr = ( SearchResult ) ne.next();
        assertEquals( "The entry returned should be the entry added earlier.", dn + "," + "dc=Apache,dc=Org", sr.getName() );

        attributes = sr.getAttributes();
        attribute = attributes.get( "description" );

        assertEquals( "The description attribute should contain the new value.", description, attribute.get() );
        assertFalse( "Search should return no more entries.", ne.hasMore() );
        System.out.println( "Modify operation successfull" );
    }


}
